#include <irrlicht.h>
#include <chrono>
#include <thread>
#include <string>

class CMainClass : public irr::IEventReceiver {
  private:
    irr::IrrlichtDevice *m_pDevice;
    irr::video::IVideoDriver  *m_pDrv;
    irr::gui::IGUIEnvironment *m_pGui;

    irr::gui::IGUITreeView *m_pTree;
    irr::gui::IGUITreeView *m_pOther;
    irr::gui::IGUICheckBox *m_pAllow;
    irr::gui::IGUIListBox  *m_pLog;
    
    void addTreeNodes(irr::gui::IGUITreeViewNode *a_pParent, int a_iLevel, int a_iParent, int &a_iCount, const std::wstring &a_sText) {
      for (int i = 0; i < 2; i++) {
        std::wstring l_sText = a_sText + L" " + std::to_wstring(++a_iCount);
        irr::gui::IGUITreeViewNode *l_pNode = a_pParent->addChildBack(l_sText.c_str(), 0, -1, -1);
        if (a_iLevel < 3) 
          addTreeNodes(l_pNode, a_iLevel + 1, i, a_iCount, a_sText);

        l_pNode->setExpanded(true);
        l_pNode->setDragDropFlag(irr::gui::EGNDF_ALLOW_ALL);
      }
    }

    void addLogMessage(const std::wstring &a_sMessage) {
      if (m_pLog != nullptr) {
        m_pLog->addItem(a_sMessage.c_str());
        m_pLog->setSelected(m_pLog->getIcon(m_pLog->getItemCount() - 1));
      }
    }

  public:
    CMainClass() : m_pDevice(nullptr), m_pDrv(nullptr), m_pGui(nullptr), m_pTree(nullptr), m_pOther(nullptr), m_pAllow(nullptr), m_pLog(nullptr) {
      m_pDevice = irr::createDevice(irr::video::EDT_OPENGL, irr::core::dimension2du(1280, 1024));

      if (m_pDevice) {
        m_pDevice->setWindowCaption(L"TreeView Test");
        m_pDevice->setEventReceiver(this);

        m_pDrv = m_pDevice->getVideoDriver();
        m_pGui = m_pDevice->getGUIEnvironment();

        irr::gui::IGUIFont *l_pFont = m_pGui->getFont("font.png");
        if (l_pFont != nullptr)
          m_pGui->getSkin()->setFont(l_pFont);
        else
          printf("Oops.");

        irr::core::dimension2du l_cFont = m_pGui->getSkin()->getFont()->getDimension(L" Allow Drag / Drop");

        m_pTree  = m_pGui->addTreeView(irr::core::recti(5, 5 + l_cFont.Height, 700, 700));
        m_pAllow = m_pGui->addCheckBox(false, irr::core::recti(5, 5, 5 + l_cFont.Width + 2 * l_cFont.Height, 5 + l_cFont.Height));
        m_pAllow->setText(L"Allow Drag / Drop");

        m_pOther = m_pGui->addTreeView(irr::core::recti(705, 5 + l_cFont.Height, 1275, 700));
        m_pOther->setDragDropFlags(irr::gui::EGTDF_ADD_AS_CHILD | irr::gui::EGTDF_ADD_AFTER | irr::gui::EGTDF_ADD_BEFORE);

        irr::gui::IGUITreeViewNode *l_pOtherRoot = m_pOther->getRoot()->addChildBack(L"Drag and drop between trees doesn't work!");
        l_pOtherRoot->setDragDropFlag(0);

        m_pLog = m_pGui->addListBox(irr::core::recti(5, 705, 1275, 1019));
        m_pLog->setAutoScrollEnabled(true);

        int i = 0;

        m_pTree->getRoot()->addChildBack(L"Cannot Drag")->setDragDropFlag(irr::gui::EGNDF_DROP_ALL);
        m_pTree->getRoot()->addChildBack(L"Cannot Drop")->setDragDropFlag(irr::gui::EGNDF_DRAG_NODE | irr::gui::EGNDF_DROP_INSERT_BEFORE | irr::gui::EGNDF_DROP_INSERT_AFTER);

        irr::gui::IGUITreeViewNode *l_pRoot = m_pTree->getRoot()->addChildBack(L"Root Node");

        addTreeNodes(l_pRoot, 0, -1, i, L"Treenode");
        l_pRoot->setExpanded(true);
        l_pRoot->setDragDropFlag(irr::gui::EGNDF_DROP_ADD_CHILD);

        i = 0;
        addTreeNodes(l_pOtherRoot, 0, -1, i, L"OtherTreeNode");
      }
    }

    bool run() {
      if (m_pDevice->run()) {
        m_pDrv->beginScene(true, true, irr::video::SColor(255, 224, 224, 224));
        m_pGui->drawAll();
        m_pDrv->endScene();

        return true;
      }
      else return false;
    }

    virtual bool OnEvent(const irr::SEvent &a_cEvent) override {
      bool l_bRet = false;

      if (a_cEvent.EventType == irr::EET_MOUSE_INPUT_EVENT) {
        if (a_cEvent.MouseInput.Event == irr::EMIE_MOUSE_MOVED) {
          // irr::gui::IGUITreeViewNode *l_pNode = m_pTree->getNodeAt(irr::core::position2di(a_cEvent.MouseInput.X, a_cEvent.MouseInput.Y));
          // if (l_pNode != nullptr)
          //   printf("==> %ls\n", l_pNode->getText());
          // else
          //   printf("==> null\n");
        }

        irr::gui::EGET_TREEVIEW_NODE_COLLAPS;
      }
      else if (a_cEvent.EventType == irr::EET_GUI_EVENT) {
        if (a_cEvent.GUIEvent.EventType == irr::gui::EGET_TREEVIEW_NODE_DRAG) {
          irr::gui::IGUITreeView *l_pTree = reinterpret_cast<irr::gui::IGUITreeView *>(a_cEvent.GUIEvent.Caller);

          if (l_pTree->getDragDropInformation() != nullptr) {
            std::wstring l_sDrag = l_pTree->getDragDropInformation()->DraggedNode->getText();
            std::wstring s = L"Tree node \"" + l_sDrag + L"\" dragged.";
            addLogMessage(s);
          }
        }
        else if (a_cEvent.GUIEvent.EventType == irr::gui::EGET_TREEVIEW_NODE_DROP) {
          irr::gui::IGUITreeView *l_pTree = reinterpret_cast<irr::gui::IGUITreeView *>(a_cEvent.GUIEvent.Caller);

          if (l_pTree->getDragDropInformation() != nullptr) {
            std::wstring l_sDrag = l_pTree->getDragDropInformation()->DraggedNode->getText();
            std::wstring l_sDrop = l_pTree->getDragDropInformation()->DropNode   ->getText();
            std::wstring l_sHow  = L""; 

            switch (l_pTree->getDragDropInformation()->DragDropState) {
              case irr::gui::EGTDF_ADD_AFTER   : l_sHow = L"added after"   ; break;
              case irr::gui::EGTDF_ADD_BEFORE  : l_sHow = L"added before"  ; break;
              case irr::gui::EGTDF_ADD_AS_CHILD: l_sHow = L"added as child"; break;
              default:
                l_sHow = L"not added";
                break;
            }

            addLogMessage(std::wstring(L"Node \"") + l_sDrag + std::wstring(L"\" ") + l_sHow + std::wstring(L" \"") + l_sDrop);
          }
        }
        else if (a_cEvent.GUIEvent.EventType == irr::gui::EGET_TREEVIEW_DRAG_CANCELED) {
          addLogMessage(L"Dragging canceled");
        }
        else if (a_cEvent.GUIEvent.EventType == irr::gui::EGET_TREEVIEW_NODE_WILL_DROP) {
          irr::gui::IGUITreeView *l_pTree = reinterpret_cast<irr::gui::IGUITreeView *>(a_cEvent.GUIEvent.Caller);

          if (l_pTree->getDragDropInformation() != nullptr) {
            std::wstring l_sDrag = l_pTree->getDragDropInformation()->DraggedNode->getText();
            std::wstring l_sDrop = l_pTree->getDragDropInformation()->DropNode   ->getText();
            std::wstring l_sHow  = L"";

            switch (l_pTree->getDragDropInformation()->DragDropState) {
              case irr::gui::EGTDF_ADD_AFTER   : l_sHow = L"will be added after"   ; break;
              case irr::gui::EGTDF_ADD_BEFORE  : l_sHow = L"will be added before"  ; break;
              case irr::gui::EGTDF_ADD_AS_CHILD: l_sHow = L"will be added as child"; break;
              default:
                l_sHow = L"will not added";
                break;
            }

            addLogMessage(std::wstring(L"Node \"") + l_sDrag + std::wstring(L"\" ") + l_sHow + std::wstring(L" \"") + l_sDrop);
          }
        }
        if (a_cEvent.GUIEvent.EventType == irr::gui::EGET_CHECKBOX_CHANGED) {
          if (a_cEvent.GUIEvent.Caller == m_pAllow) {
            m_pTree->setDragDropFlags(m_pAllow->isChecked() ? irr::gui::EGTDF_ALLOW_DRAG_DROP : irr::gui::EGTDF_NONE);
            addLogMessage(L"Drag / Drop on the left tree " + std::wstring(m_pAllow->isChecked() ? L"activated" : L"disabled"));
          }
        }
      }

      return l_bRet;
    }
};

int main(void) {
  CMainClass l_cMain;

  std::chrono::steady_clock::time_point l_cNextStep = std::chrono::steady_clock::now();
  while (l_cMain.run()) {
    l_cNextStep = l_cNextStep + std::chrono::duration<int, std::ratio<1, 1000>>(16);
    std::this_thread::sleep_until(l_cNextStep);
  }
}